/* =============================================================================
 *
 * Digital_Scope_Demo.c
 *
 * This file contains the code for the digital scope demo task.
 *
 * v1.0	YFS	Initial release. 
 *
 * =============================================================================
*/

/*
* Copyright (2014), Cypress Semiconductor Corporation. All Rights Reserved.
*
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express written permission of Cypress.
*
* Disclaimer: CYPRESS MAKES NO WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, WITH 
* REGARD TO THIS MATERIAL, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
* Cypress reserves the right to make changes without further notice to the 
* materials described herein. Cypress does not assume any liability arising out 
* of the application or use of any product or circuit described herein. Cypress 
* does not authorize its products for use as critical components in life-support 
* systems where a malfunction or failure may reasonably be expected to result in 
* significant injury to the user. The inclusion of Cypress' product in a life-
* support systems application implies that the manufacturer assumes all risk of 
* such use and in doing so indemnifies Cypress against all charges. 
*
* Use of this Software may be limited by and subject to the applicable Cypress
* software license agreement. 
*/

/* Include FreeRTOS APIs and defines */
#include <FreeRTOS.h>
#include <task.h>
#include <queue.h>
#include <event_groups.h>

/* Include PSoC system and component APIs and defines */
#include <project.h>

/* Include application function declarations and defines */
#include <utils.h>
#include <App_Defs.h>
#include <Task_Defs.h>

/*
 * Function:		Digital_Scope_Demo_Task
 *
 * This function runs as a FreeRTOS task with a medium priority. It is a
 * demonstration of a simple digital signal scope. SW2 button presses are
 * displayed as a moving waveform on the LCD. The Capsense slider is used to
 * vary the speed of the scope.
 *
 * Execution of this task is controlled by an event group. After creating
 * this task, the Main_Task starts the demo by setting the bit in Demo_Events
 * that corresponds to this demo. When the user signals a desire to end the
 * demo, it stops the LCD and then waits on the event bit again. This leaves
 * the task ready to be re-executed at a later date.
 *
 *			Start task
 *			Initialize LCD buffer
 *			Wait on its event bit (wait for user to request demo)
 *			Clear LCD and print demo text
 *			Print instructions
 *		 -> Read the state of the switch and write approprate char to buffer
 *		|	Print out the buffer, oldest char first, so movement is R-to-L
 *		|	Pend (non-blocking) on the capsense queue
 *		|	If the slider is active change the LCD update rate (delay)
 *		|	Delay
 *		|   Check if SW3 is pressed. If so...
 *		|   	Flush the CapSense queue
 *      |       Set the Main_Task event bit (make it ready to run)
 *      |   	Wait on its event bit (sleep)
 *      |   	Clear LCD, print demo text, prep the buffer, debounce SW2
 *		 -- Loop
 *
 * Globals:		Uses Demo_Events, Main_Task_Event and CapSense_Monitor_Q
 *
 * Parameters:	pvParameters (demo index number)
 *
 * Return:		None (infinite loop)
 */
 
void Digital_Scope_Demo_Task( void *pvParameters )
{
	int demoNumber = (int)pvParameters;     /* Passed from xTaskCreate */
    
    BaseType_t queueDataReceived;           /* Return from queue API */
	
	int LCD_update_delay = SCOPE_DELAY_DEFAULT;	/* Time between LCD updates */
	
	CS_PACKET capsensepacket;	/* Hold a message from CapSense Q */
	
    int state = LO;				/* Start by assuming button is not pressed */

	int pressed;				/* Use this to flag the switch being pressed */

	/* Declare the buffer, its index and a loop counter to clear/print it */
	char buffer[LCD_WIDTH];
	int index = 0;
	int i;
	
	/* Clear the buffer - set all chars to 'space' */
	for( i=0; i<LCD_WIDTH; i++ )
	{
		buffer[i] = ' ';
	}
	
	/* Pend on own task semaphore so that Main_Task can run */
    xEventGroupWaitBits(
        Demo_Events,            /* Wait for this demo to be requested */
        1 << demoNumber,        /* Event to wait - each demo owns a bit in the group */
        pdTRUE,                 /* Clear the event on exit */
        pdFALSE,                /* Do not wait for all bits (irrelevant) */
        WAIT_FOREVER );
	
	/*
	We woke up so Main_Task has set the event and it is time to run the demo.
	Prepare the LCD to show the demo results.
	*/
	LCD_ClearDisplay();
	LCD_PrintString( DIGITAL_SCOPE_DEMO_TASK );
	
	CyDelay( 250 );	/* Debounce (250ms) press that started the demo */
	
	/* Give the user some instructions */
	scroll( 1, "Apply signal via SW2. Use slider to change speed." );
	
    for( ; ; )		/* Demo main loop (forever) */
    {
		/* Read SW2 - reverse the logic (!) so pressed (grounded) means HI */
		pressed = ! P6_1_Read();
		
		/* Pick the character that represents the state of the signal */
        if( !pressed && state==LO )
		{
			buffer[index] = CHAR_LO;		/* Signal is low and stays low */
		}
		else if( pressed && state==HI )
		{
			buffer[index] = CHAR_HI;		/* Signal is high and stays high */
		}		
		else if( pressed && state==LO )
		{
			buffer[index] = CHAR_RISE;		/* Signal is low and goes high */
			state = HI;						/* New state */
		}
		else if( !pressed && state==HI )
		{
			buffer[index] = CHAR_FALL;		/* Signal is high and goes low */
			state = LO;						/* New state */
		}
		
		/*
		Print out the buffer, oldest char first, so movement is R-to-L.
		
		The following moves the index ahead of the latest character (points to
		the oldest) and then prints out LCD_WIDTH characters, ending with the
		newest. The character to print is accessed from the index, plus an
		increment from 0 to 16, modulo LCD_WIDTH (to wrap the buffer index).
		
		First increment index and, if at the end of the buffer, wrap back to 0.
		*/
		index++;
		if( index == LCD_WIDTH )
		{
			index = 0;
		}
		
		/* Next print out the LCD_WIDTH chars */
		LCD_Position( 1, 0 );
		for( i=0; i<LCD_WIDTH; i++ )
		{
			LCD_PutChar( buffer[(index + i) % LCD_WIDTH] );
		}
		
		/* Collect the CapSense activity from the message queue */
        queueDataReceived = xQueueReceive(
            CapSense_Monitor_Q,
            &capsensepacket.result,
            DO_NOT_WAIT_ON_QUEUE );
        
        if( queueDataReceived == pdTRUE )
        {
            /* Calculate new update rate from the slider value */
				if( capsensepacket.slider )
				{
					/* Delay is between 100 and 400 (100+3*100) ms */
					LCD_update_delay = SCOPE_DELAY_MIN + 
								capsensepacket.sliderval * SCOPE_DELAY_UNITS;
				}
				/* Note: Buttons presses are ignored */            
        }
                		
		/* Use delay call to change the speed of the scope */
		vTaskDelay( LCD_update_delay );
		
		/* Check to see if SW3 was pressed - this signals the end of the demo */
		if( ! P15_5_Read() )		/* Note: 0 (ground) means SW3 was pressed */
		{	
			/* Flush any CapSense data from queue - it is no longer relvant */
			xQueueReset( CapSense_Monitor_Q );
			
			/* Post to the MainTask semaphore to make it ready to run */
			xEventGroupSetBits( Main_Task_Event, MAIN_TASK_EVENT_NUM );

        	/* Pend on own task semaphore so that MainTask can run */
            xEventGroupWaitBits(
                Demo_Events,            /* Wait for this demo to be requested */
                1 << demoNumber,        /* Event to wait - each demo owns a bit */
                pdTRUE,                 /* Clear the event on exit */
                pdFALSE,                /* Do not wait for all bits (irrelevant) */
                WAIT_FOREVER );
			
			/*
			MainTask re-set the event bit so it is time to run the demo again.
			Prepare the LCD to show the demo results. Debounce SW2.
			*/	
			LCD_ClearDisplay();
			LCD_PrintString( DIGITAL_SCOPE_DEMO_TASK );
			
			/* Reset and clear buffer - so demo restarts from RHS of screen */
			index = 0;

			for( i=0; i<LCD_WIDTH; i++ )
			{
				buffer[i] = ' ';
			}
			
			CyDelay( 250 );	/* Debounce (250ms) press that started the demo */
            
		} /* if( ! P15_5_Read() ) */	
		
    } /* for( ; ; ) */
	
} /* Digital_Scope_Demo_Task() */
